import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { usersApi } from "@/lib/api";
import { useLocation } from "wouter";
import { useState } from "react";
import { Loader2, Plus, Edit, Trash2, ArrowRight, UserPlus } from "lucide-react";
import { toast } from "sonner";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";

export default function Users() {
  const { user, isAuthenticated } = useAuth();
  const [, setLocation] = useLocation();
  const queryClient = useQueryClient();

  const { data: users, isLoading } = useQuery({
    queryKey: ['users'],
    queryFn: async () => {
      const response = await usersApi.list();
      return response.data;
    },
    enabled: isAuthenticated && user?.role === "admin",
  });

  const [showCreateDialog, setShowCreateDialog] = useState(false);
  const [showEditDialog, setShowEditDialog] = useState<number | null>(null);
  const [showPasswordDialog, setShowPasswordDialog] = useState<number | null>(null);
  const [showDeleteDialog, setShowDeleteDialog] = useState<number | null>(null);

  const [createForm, setCreateForm] = useState({
    username: "",
    password: "",
    name: "",
    email: "",
    role: "user" as "user" | "admin",
  });

  const [editForm, setEditForm] = useState({
    name: "",
    email: "",
    role: "user" as "user" | "admin",
  });

  const [passwordForm, setPasswordForm] = useState({
    newPassword: "",
    confirmPassword: "",
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => usersApi.create(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['users'] });
      toast.success("کاربر با موفقیت ایجاد شد");
      setShowCreateDialog(false);
      setCreateForm({ username: "", password: "", name: "", email: "", role: "user" });
    },
    onError: (error: any) => {
      const message = error.response?.data?.message || "خطا در ایجاد کاربر";
      toast.error(message);
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) => usersApi.update(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['users'] });
      toast.success("کاربر با موفقیت به‌روزرسانی شد");
      setShowEditDialog(null);
    },
    onError: (error: any) => {
      const message = error.response?.data?.message || "خطا در به‌روزرسانی کاربر";
      toast.error(message);
    },
  });

  const updatePasswordMutation = useMutation({
    mutationFn: ({ id, newPassword }: { id: number; newPassword: string }) =>
      usersApi.updatePassword(id, newPassword),
    onSuccess: () => {
      toast.success("رمز عبور با موفقیت تغییر کرد");
      setShowPasswordDialog(null);
      setPasswordForm({ newPassword: "", confirmPassword: "" });
    },
    onError: (error: any) => {
      const message = error.response?.data?.message || "خطا در تغییر رمز عبور";
      toast.error(message);
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => usersApi.delete(id),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['users'] });
      toast.success("کاربر با موفقیت حذف شد");
      setShowDeleteDialog(null);
    },
    onError: (error: any) => {
      const message = error.response?.data?.message || "خطا در حذف کاربر";
      toast.error(message);
    },
  });

  if (!isAuthenticated || user?.role !== "admin") {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100">
        <Card className="bg-white">
          <CardContent className="py-8 text-center">
            <p className="text-gray-500">شما دسترسی به این صفحه ندارید</p>
            <Button onClick={() => setLocation("/")} className="mt-4">
              بازگشت به صفحه اصلی
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  const handleCreate = () => {
    if (!createForm.username.trim() || !createForm.password.trim()) {
      toast.error("نام کاربری و رمز عبور الزامی است");
      return;
    }
    createMutation.mutate(createForm);
  };

  const handleEdit = (userId: number) => {
    const userToEdit = users?.find((u: any) => u.id === userId);
    if (userToEdit) {
      setEditForm({
        name: userToEdit.name || "",
        email: userToEdit.email || "",
        role: userToEdit.role,
      });
      setShowEditDialog(userId);
    }
  };

  const handleUpdate = () => {
    if (showEditDialog) {
      updateMutation.mutate({
        id: showEditDialog,
        data: editForm,
      });
    }
  };

  const handleChangePassword = () => {
    if (!passwordForm.newPassword || passwordForm.newPassword.length < 6) {
      toast.error("رمز عبور باید حداقل 6 کاراکتر باشد");
      return;
    }
    if (passwordForm.newPassword !== passwordForm.confirmPassword) {
      toast.error("رمز عبور و تکرار آن مطابقت ندارند");
      return;
    }
    if (showPasswordDialog) {
      updatePasswordMutation.mutate({
        id: showPasswordDialog,
        newPassword: passwordForm.newPassword,
      });
    }
  };

  const handleDelete = () => {
    if (showDeleteDialog) {
      deleteMutation.mutate(showDeleteDialog);
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-8" dir="rtl">
      <div className="max-w-7xl mx-auto px-4">
        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">مدیریت کاربران</h1>
            <p className="text-gray-600 mt-2">مدیریت کاربران سیستم</p>
          </div>
          <div className="flex gap-2">
            <Button variant="outline" onClick={() => setLocation("/")} className="gap-2">
              <ArrowRight className="w-4 h-4" />
              بازگشت
            </Button>
            <Button onClick={() => setShowCreateDialog(true)} className="gap-2 bg-indigo-600 hover:bg-indigo-700">
              <UserPlus className="w-4 h-4" />
              کاربر جدید
            </Button>
          </div>
        </div>

        <Card className="bg-white">
          <CardHeader>
            <CardTitle>لیست کاربران</CardTitle>
            <CardDescription>{users?.length || 0} کاربر در سیستم</CardDescription>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="flex justify-center py-8">
                <Loader2 className="w-6 h-6 animate-spin text-indigo-600" />
              </div>
            ) : users && users.length > 0 ? (
              <div className="overflow-x-auto">
                <table className="w-full text-sm">
                  <thead className="border-b border-gray-200 bg-gray-50">
                    <tr className="text-right">
                      <th className="px-4 py-3 font-semibold text-gray-700">نام کاربری</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">نام</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">ایمیل</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">نقش</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">آخرین ورود</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">عملیات</th>
                    </tr>
                  </thead>
                  <tbody>
                    {users.map((u: any) => (
                      <tr key={u.id} className="border-b border-gray-100 hover:bg-gray-50">
                        <td className="px-4 py-3 font-mono">{u.username}</td>
                        <td className="px-4 py-3">{u.name || "-"}</td>
                        <td className="px-4 py-3">{u.email || "-"}</td>
                        <td className="px-4 py-3">
                          <span
                            className={`px-2 py-1 rounded-full text-xs font-semibold ${
                              u.role === "admin"
                                ? "bg-red-100 text-red-800"
                                : "bg-blue-100 text-blue-800"
                            }`}
                          >
                            {u.role === "admin" ? "مدیر" : "کاربر"}
                          </span>
                        </td>
                        <td className="px-4 py-3 text-gray-600">
                          {u.last_signed_in
                            ? new Date(u.last_signed_in).toLocaleDateString("fa-IR")
                            : "-"}
                        </td>
                        <td className="px-4 py-3">
                          <div className="flex gap-2">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => handleEdit(u.id)}
                              className="gap-1"
                            >
                              <Edit className="w-3 h-3" />
                              ویرایش
                            </Button>
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => {
                                setPasswordForm({ newPassword: "", confirmPassword: "" });
                                setShowPasswordDialog(u.id);
                              }}
                              className="gap-1"
                            >
                              رمز عبور
                            </Button>
                            {u.id !== user?.id && (
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => setShowDeleteDialog(u.id)}
                                className="gap-1"
                              >
                                <Trash2 className="w-3 h-3" />
                                حذف
                              </Button>
                            )}
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            ) : (
              <div className="text-center py-8">
                <p className="text-gray-500">هیچ کاربری یافت نشد</p>
              </div>
            )}
          </CardContent>
        </Card>
      </div>

      {/* Dialog ایجاد کاربر */}
      <Dialog open={showCreateDialog} onOpenChange={setShowCreateDialog}>
        <DialogContent className="sm:max-w-[425px]">
          <DialogHeader>
            <DialogTitle>ایجاد کاربر جدید</DialogTitle>
            <DialogDescription>اطلاعات کاربر جدید را وارد کنید</DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div>
              <Label htmlFor="create-username">نام کاربری *</Label>
              <Input
                id="create-username"
                value={createForm.username}
                onChange={(e) => setCreateForm({ ...createForm, username: e.target.value })}
                placeholder="نام کاربری"
              />
            </div>
            <div>
              <Label htmlFor="create-password">رمز عبور *</Label>
              <Input
                id="create-password"
                type="password"
                value={createForm.password}
                onChange={(e) => setCreateForm({ ...createForm, password: e.target.value })}
                placeholder="حداقل 6 کاراکتر"
              />
            </div>
            <div>
              <Label htmlFor="create-name">نام</Label>
              <Input
                id="create-name"
                value={createForm.name}
                onChange={(e) => setCreateForm({ ...createForm, name: e.target.value })}
                placeholder="نام"
              />
            </div>
            <div>
              <Label htmlFor="create-email">ایمیل</Label>
              <Input
                id="create-email"
                type="email"
                value={createForm.email}
                onChange={(e) => setCreateForm({ ...createForm, email: e.target.value })}
                placeholder="email@example.com"
              />
            </div>
            <div>
              <Label htmlFor="create-role">نقش</Label>
              <Select
                value={createForm.role}
                onValueChange={(value) => setCreateForm({ ...createForm, role: value as "user" | "admin" })}
              >
                <SelectTrigger id="create-role">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="user">کاربر</SelectItem>
                  <SelectItem value="admin">مدیر</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowCreateDialog(false)}>
              انصراف
            </Button>
            <Button
              onClick={handleCreate}
              disabled={createMutation.isPending}
              className="bg-indigo-600 hover:bg-indigo-700"
            >
              {createMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                  در حال ایجاد...
                </>
              ) : (
                "ایجاد"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Dialog ویرایش کاربر */}
      <Dialog open={showEditDialog !== null} onOpenChange={(open) => !open && setShowEditDialog(null)}>
        <DialogContent className="sm:max-w-[425px]">
          <DialogHeader>
            <DialogTitle>ویرایش کاربر</DialogTitle>
            <DialogDescription>اطلاعات کاربر را ویرایش کنید</DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div>
              <Label htmlFor="edit-name">نام</Label>
              <Input
                id="edit-name"
                value={editForm.name}
                onChange={(e) => setEditForm({ ...editForm, name: e.target.value })}
                placeholder="نام"
              />
            </div>
            <div>
              <Label htmlFor="edit-email">ایمیل</Label>
              <Input
                id="edit-email"
                type="email"
                value={editForm.email}
                onChange={(e) => setEditForm({ ...editForm, email: e.target.value })}
                placeholder="email@example.com"
              />
            </div>
            <div>
              <Label htmlFor="edit-role">نقش</Label>
              <Select
                value={editForm.role}
                onValueChange={(value) => setEditForm({ ...editForm, role: value as "user" | "admin" })}
              >
                <SelectTrigger id="edit-role">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="user">کاربر</SelectItem>
                  <SelectItem value="admin">مدیر</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowEditDialog(null)}>
              انصراف
            </Button>
            <Button
              onClick={handleUpdate}
              disabled={updateMutation.isPending}
              className="bg-indigo-600 hover:bg-indigo-700"
            >
              {updateMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                  در حال ذخیره...
                </>
              ) : (
                "ذخیره"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Dialog تغییر رمز عبور */}
      <Dialog open={showPasswordDialog !== null} onOpenChange={(open) => !open && setShowPasswordDialog(null)}>
        <DialogContent className="sm:max-w-[425px]">
          <DialogHeader>
            <DialogTitle>تغییر رمز عبور</DialogTitle>
            <DialogDescription>رمز عبور جدید را وارد کنید</DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div>
              <Label htmlFor="new-password">رمز عبور جدید *</Label>
              <Input
                id="new-password"
                type="password"
                value={passwordForm.newPassword}
                onChange={(e) => setPasswordForm({ ...passwordForm, newPassword: e.target.value })}
                placeholder="حداقل 6 کاراکتر"
              />
            </div>
            <div>
              <Label htmlFor="confirm-password">تکرار رمز عبور *</Label>
              <Input
                id="confirm-password"
                type="password"
                value={passwordForm.confirmPassword}
                onChange={(e) => setPasswordForm({ ...passwordForm, confirmPassword: e.target.value })}
                placeholder="تکرار رمز عبور"
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowPasswordDialog(null)}>
              انصراف
            </Button>
            <Button
              onClick={handleChangePassword}
              disabled={updatePasswordMutation.isPending}
              className="bg-indigo-600 hover:bg-indigo-700"
            >
              {updatePasswordMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                  در حال تغییر...
                </>
              ) : (
                "تغییر رمز عبور"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Dialog حذف کاربر */}
      <Dialog open={showDeleteDialog !== null} onOpenChange={(open) => !open && setShowDeleteDialog(null)}>
        <DialogContent className="sm:max-w-[425px]">
          <DialogHeader>
            <DialogTitle>حذف کاربر</DialogTitle>
            <DialogDescription>آیا از حذف این کاربر اطمینان دارید؟</DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowDeleteDialog(null)}>
              انصراف
            </Button>
            <Button
              variant="destructive"
              onClick={handleDelete}
              disabled={deleteMutation.isPending}
            >
              {deleteMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                  در حال حذف...
                </>
              ) : (
                "حذف"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}

