import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { reportsApi } from "@/lib/api";
import { useLocation } from "wouter";
import { useState } from "react";
import { Loader2, ArrowRight, Download, Eye } from "lucide-react";
import { toast } from "sonner";
import { useQuery } from "@tanstack/react-query";

export default function Reports() {
  const { isAuthenticated } = useAuth();
  const [, setLocation] = useLocation();

  const [filters, setFilters] = useState({
    reportType: "monthly" as "daily" | "monthly" | "yearly",
    startDate: new Date(new Date().setMonth(new Date().getMonth() - 1)).toISOString().split("T")[0],
    endDate: new Date().toISOString().split("T")[0],
  });

  const [useDateRange, setUseDateRange] = useState(false);

  const { data: reports, isLoading: reportsLoading } = useQuery({
    queryKey: ['reports'],
    queryFn: async () => {
      const response = await reportsApi.list();
      return response.data;
    },
    enabled: isAuthenticated && !useDateRange,
  });

  const { data: dateRangeReports, isLoading: dateRangeLoading, refetch: refetchDateRange } = useQuery({
    queryKey: ['reports', 'date-range', filters.startDate, filters.endDate],
    queryFn: async () => {
      const response = await reportsApi.getByDateRange(filters.startDate, filters.endDate);
      return response.data;
    },
    enabled: false,
  });

  if (!isAuthenticated) {
    return null;
  }

  const handleSearch = async () => {
    if (!filters.startDate || !filters.endDate) {
      toast.error("لطفاً تاریخ شروع و پایان را مشخص کنید");
      return;
    }

    setUseDateRange(true);
    await refetchDateRange();
  };

  const displayReports = useDateRange ? (dateRangeReports || []) : (reports || []);

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-8" dir="rtl">
      <div className="max-w-6xl mx-auto px-4">
        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">گزارش‌های حسابداری</h1>
            <p className="text-gray-600 mt-2">مشاهده و مدیریت گزارش‌های درآمدی</p>
          </div>
          <Button
            variant="outline"
            onClick={() => setLocation("/")}
            className="gap-2"
          >
            <ArrowRight className="w-4 h-4" />
            بازگشت
          </Button>
        </div>

        <Card className="bg-white mb-6">
          <CardHeader>
            <CardTitle>فیلتر گزارش‌ها</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div>
                <Label htmlFor="reportType">نوع گزارش</Label>
                <Select
                  value={filters.reportType}
                  onValueChange={(value) =>
                    setFilters((prev) => ({
                      ...prev,
                      reportType: value as "daily" | "monthly" | "yearly",
                    }))
                  }
                >
                  <SelectTrigger id="reportType">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="daily">روزانه</SelectItem>
                    <SelectItem value="monthly">ماهانه</SelectItem>
                    <SelectItem value="yearly">سالانه</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="startDate">تاریخ شروع</Label>
                <Input
                  id="startDate"
                  type="date"
                  value={filters.startDate}
                  onChange={(e) =>
                    setFilters((prev) => ({
                      ...prev,
                      startDate: e.target.value,
                    }))
                  }
                />
              </div>

              <div>
                <Label htmlFor="endDate">تاریخ پایان</Label>
                <Input
                  id="endDate"
                  type="date"
                  value={filters.endDate}
                  onChange={(e) =>
                    setFilters((prev) => ({
                      ...prev,
                      endDate: e.target.value,
                    }))
                  }
                />
              </div>

              <div className="flex items-end">
                <Button
                  onClick={handleSearch}
                  disabled={dateRangeLoading}
                  className="w-full bg-indigo-600 hover:bg-indigo-700"
                >
                  {dateRangeLoading ? (
                    <>
                      <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                      جستجو...
                    </>
                  ) : (
                    "جستجو"
                  )}
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-white">
          <CardHeader>
            <CardTitle>گزارش‌ها</CardTitle>
            <CardDescription>
              {displayReports.length} گزارش یافت شد
            </CardDescription>
          </CardHeader>
          <CardContent>
            {reportsLoading || dateRangeLoading ? (
              <div className="flex justify-center py-8">
                <Loader2 className="w-6 h-6 animate-spin text-indigo-600" />
              </div>
            ) : displayReports.length > 0 ? (
              <div className="overflow-x-auto">
                <table className="w-full text-sm">
                  <thead className="border-b border-gray-200 bg-gray-50">
                    <tr className="text-right">
                      <th className="px-4 py-3 font-semibold text-gray-700">نوع گزارش</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">تاریخ</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">تعداد آگهی</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">مجموع درآمد</th>
                      <th className="px-4 py-3 font-semibold text-gray-700">عملیات</th>
                    </tr>
                  </thead>
                  <tbody>
                    {displayReports.map((report: any) => {
                      const reportTypeLabel: Record<string, string> = {
                        daily: "روزانه",
                        monthly: "ماهانه",
                        yearly: "سالانه",
                      };
                      const typeLabel = reportTypeLabel[report.report_type] || report.report_type;

                      return (
                        <tr
                          key={report.id}
                          className="border-b border-gray-100 hover:bg-gray-50"
                        >
                          <td className="px-4 py-3 font-semibold text-gray-900">
                            {typeLabel}
                          </td>
                          <td className="px-4 py-3 text-gray-600">{report.report_date}</td>
                          <td className="px-4 py-3 text-gray-600">
                            {report.advertisement_count}
                          </td>
                          <td className="px-4 py-3 font-semibold text-indigo-600">
                            {report.total_amount.toLocaleString('fa-IR')} ریال
                          </td>
                          <td className="px-4 py-3">
                            <div className="flex gap-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => {
                                  const newWindow = window.open("", "", "width=1000,height=800");
                                  if (newWindow && report.report_content) {
                                    newWindow.document.write(report.report_content);
                                    newWindow.document.close();
                                  }
                                }}
                              >
                                <Eye className="w-4 h-4" />
                              </Button>
                              {report.pdf_url && (
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => {
                                    window.open(report.pdf_url, "_blank");
                                  }}
                                >
                                  <Download className="w-4 h-4" />
                                </Button>
                              )}
                            </div>
                          </td>
                        </tr>
                      );
                    })}
                  </tbody>
                </table>
              </div>
            ) : (
              <div className="text-center py-8">
                <p className="text-gray-500">هیچ گزارشی یافت نشد</p>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}

