import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { advertisementsApi, receiptsApi } from "@/lib/api";
import { useLocation } from "wouter";
import { Loader2, Download, Printer, ArrowRight } from "lucide-react";
import { toast } from "sonner";
import { useQuery } from "@tanstack/react-query";
import { gregorianToJalali } from "@/lib/dateUtils";

interface ReceiptViewProps {
  advertisementId: number;
}

export default function ReceiptView({ advertisementId }: ReceiptViewProps) {
  const { isAuthenticated } = useAuth();
  const [, setLocation] = useLocation();

  const { data: advertisement, isLoading: adLoading } = useQuery({
    queryKey: ['advertisements', advertisementId],
    queryFn: async () => {
      const response = await advertisementsApi.getById(advertisementId);
      return response.data;
    },
    enabled: isAuthenticated,
  });

  const { data: receiptData, isLoading: datesLoading } = useQuery({
    queryKey: ['receipts', 'advertisement', advertisementId],
    queryFn: async () => {
      const response = await receiptsApi.getByAdvertisement(advertisementId);
      return response.data;
    },
    enabled: isAuthenticated && !!advertisement,
  });

  if (!isAuthenticated) {
    return null;
  }

  if (adLoading || datesLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100">
        <Loader2 className="w-12 h-12 animate-spin text-indigo-600" />
      </div>
    );
  }

  if (!advertisement) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-8">
        <div className="max-w-4xl mx-auto px-4">
          <Button variant="outline" onClick={() => setLocation("/")} className="mb-6">
            <ArrowRight className="w-4 h-4 ml-2" />
            بازگشت
          </Button>
          <Card className="bg-white">
            <CardContent className="py-8 text-center">
              <p className="text-gray-500">آگهی مورد نظر یافت نشد</p>
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  const printDatesArray = receiptData?.print_dates || [];
  const issueDateJalali = advertisement.issue_date ? gregorianToJalali(advertisement.issue_date) : advertisement.issue_date;

  const handlePrint = () => {
    window.print();
  };

  const handleDownloadPDF = async () => {
    toast.promise(
      new Promise((resolve, reject) => {
        try {
          const element = document.getElementById("receipt-content");
          if (element) {
            const printWindow = window.open("", "", "width=800,height=600");
            if (printWindow) {
              printWindow.document.write(element.innerHTML);
              printWindow.document.close();
              setTimeout(() => {
                printWindow.print();
                resolve(true);
              }, 250);
            } else {
              reject(new Error("نتوانستم پنجره چاپ را باز کنم"));
            }
          } else {
            reject(new Error("محتوای قبض یافت نشد"));
          }
        } catch (error) {
          reject(error);
        }
      }),
      {
        loading: "در حال آماده‌سازی...",
        success: "آماده برای چاپ یا ذخیره به عنوان PDF",
        error: (err) => `خطا: ${err instanceof Error ? err.message : "خطایی نامشخص"}`,
      }
    );
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-8" dir="rtl">
      <div className="max-w-4xl mx-auto px-4">
        <div className="flex gap-2 mb-6 flex-wrap">
          <Button
            variant="outline"
            onClick={() => setLocation(`/advertisements/${advertisementId}`)}
            className="gap-2"
          >
            <ArrowRight className="w-4 h-4" />
            بازگشت
          </Button>
          <Button
            onClick={handlePrint}
            className="bg-blue-600 hover:bg-blue-700 text-white gap-2"
          >
            <Printer className="w-4 h-4" />
            چاپ
          </Button>
          <Button
            onClick={handleDownloadPDF}
            className="bg-green-600 hover:bg-green-700 text-white gap-2"
          >
            <Download className="w-4 h-4" />
            دانلود PDF
          </Button>
        </div>

        <Card className="bg-white" id="receipt-content">
          <CardHeader className="border-b-2 border-gray-300 bg-gray-50">
            <div className="flex flex-row-reverse justify-between items-start">
              <div className="text-right">
                <CardTitle className="text-3xl font-bold">قبض حق‌الدرج آگهی</CardTitle>
                <CardDescription className="text-base mt-2">روزنامه خبر جنوب</CardDescription>
              </div>
              <div className="text-left">
                <p className="text-red-600 font-bold text-2xl mb-2">
                  {advertisement.receipt_number} :شماره قبض
                </p>
                <p className="text-gray-600">{issueDateJalali} :تاریخ</p>
              </div>
            </div>
          </CardHeader>

          <CardContent className="py-8">
            <div className="space-y-6">
              <div className="grid grid-cols-2 gap-6">
                <div className="border-l-4 border-indigo-500 pl-4 text-right">
                  <p className="text-sm text-gray-600 font-semibold mb-2">نام صاحب اگهی</p>
                  <p className="text-lg font-bold text-gray-900">{advertisement.advertiser_name}</p>
                </div>
                <div className="border-l-4 border-blue-500 pl-4 text-right">
                  <p className="text-sm text-gray-600 font-semibold mb-2">موضوع اگهی</p>
                  <p className="text-lg font-bold text-gray-900">{advertisement.subject}</p>
                </div>
              </div>

              <div className="bg-gradient-to-r from-indigo-50 to-blue-50 p-6 rounded-lg border-2 border-indigo-200">
                <div className="grid grid-cols-2 gap-6">
                  <div className="text-right">
                    <p className="text-sm text-gray-600 font-semibold mb-2">مبلغ به عدد</p>
                    <p className="text-2xl font-bold text-indigo-600">
                      {advertisement.amount_in_numbers.toLocaleString('fa-IR')}
                    </p>
                    <p className="text-xs text-gray-500 mt-1">ریال</p>
                  </div>
                  <div className="text-right">
                    <p className="text-sm text-gray-600 font-semibold mb-2">مبلغ به حروف</p>
                    <p className="text-base font-semibold text-gray-900">{advertisement.amount_in_words}</p>
                  </div>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-6">
                <div className="text-right">
                  <p className="text-sm text-gray-600 font-semibold mb-2">تعداد نوبت</p>
                  <p className="text-lg font-bold text-gray-900">{advertisement.number_of_times}</p>
                </div>
                <div className="text-right">
                  <p className="text-sm text-gray-600 font-semibold mb-2">نحوه وصول</p>
                  <p className="text-lg font-bold text-gray-900">{advertisement.payment_method}</p>
                </div>
              </div>

              {printDatesArray.length > 0 && (
                <div className="bg-blue-50 p-4 rounded-lg border border-blue-200">
                  <p className="text-sm text-gray-600 font-semibold mb-3">تاریخ‌های چاپ</p>
                  <div className="flex gap-2 flex-wrap">
                    {printDatesArray.map((date: any, index: number) => (
                      <span
                        key={index}
                        className="bg-blue-600 text-white px-4 py-2 rounded-full text-sm font-semibold"
                      >
                        {date.print_date ? gregorianToJalali(date.print_date) : date.print_date}
                      </span>
                    ))}
                  </div>
                </div>
              )}

              {advertisement.order_number && (
                <div>
                  <p className="text-sm text-gray-600 font-semibold mb-2">شماره دستور</p>
                  <p className="text-base text-gray-900 font-mono">{advertisement.order_number}</p>
                </div>
              )}

              {advertisement.signature_location && (
                <div>
                  <p className="text-sm text-gray-600 font-semibold mb-2">جای امضا</p>
                  <div className="bg-gray-50 p-4 rounded border border-gray-300 min-h-20 text-gray-700">
                    {advertisement.signature_location}
                  </div>
                </div>
              )}
            </div>

            <div className="mt-12 pt-8 border-t-2 border-gray-300">
              <div className="grid grid-cols-3 gap-4 text-center">
                <div>
                  <p className="text-sm font-semibold text-gray-700 mb-16">مهر و امضای مدیر</p>
                  <div className="border-t-2 border-gray-400 pt-2"></div>
                </div>
                <div>
                  <p className="text-sm font-semibold text-gray-700 mb-16">امضای صادرکننده</p>
                  <div className="border-t-2 border-gray-400 pt-2"></div>
                </div>
                <div>
                  <p className="text-sm font-semibold text-gray-700 mb-16">امضای دریافت‌کننده</p>
                  <div className="border-t-2 border-gray-400 pt-2"></div>
                </div>
              </div>
            </div>

            <div className="mt-8 pt-6 border-t border-gray-200 text-center text-xs text-gray-500">
              <p>این قبض نمایانگر حق‌الدرج آگهی در روزنامه خبر جنوب است.</p>
              <p className="mt-1">تاریخ صدور: {new Date().toLocaleDateString('fa-IR')}</p>
            </div>
          </CardContent>
        </Card>
      </div>

      <style>{`
        @media print {
          body {
            background: white;
            padding: 0;
            margin: 0;
          }
          .bg-gradient-to-br,
          .bg-blue-50,
          .bg-gray-50 {
            background: white !important;
          }
          button,
          .flex.gap-2.mb-6 {
            display: none !important;
          }
          .max-w-4xl {
            max-width: 100%;
          }
        }
      `}</style>
    </div>
  );
}

