import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { usersApi } from "@/lib/api";
import { useLocation } from "wouter";
import { useState } from "react";
import { Loader2, ArrowRight, Lock } from "lucide-react";
import { toast } from "sonner";
import { useMutation } from "@tanstack/react-query";

export default function ChangePassword() {
  const { isAuthenticated } = useAuth();
  const [, setLocation] = useLocation();

  const [formData, setFormData] = useState({
    currentPassword: "",
    newPassword: "",
    confirmPassword: "",
  });

  const changePasswordMutation = useMutation({
    mutationFn: (data: { currentPassword: string; newPassword: string }) =>
      usersApi.changeMyPassword(data.currentPassword, data.newPassword),
    onSuccess: () => {
      toast.success("رمز عبور با موفقیت تغییر کرد");
      setFormData({
        currentPassword: "",
        newPassword: "",
        confirmPassword: "",
      });
      setLocation("/");
    },
    onError: (error: any) => {
      const message = error.response?.data?.message || "خطا در تغییر رمز عبور";
      toast.error(message);
    },
  });

  if (!isAuthenticated) {
    return null;
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.currentPassword.trim()) {
      toast.error("رمز عبور فعلی الزامی است");
      return;
    }

    if (!formData.newPassword.trim() || formData.newPassword.length < 6) {
      toast.error("رمز عبور جدید باید حداقل 6 کاراکتر باشد");
      return;
    }

    if (formData.newPassword !== formData.confirmPassword) {
      toast.error("رمز عبور جدید و تکرار آن مطابقت ندارند");
      return;
    }

    changePasswordMutation.mutate({
      currentPassword: formData.currentPassword,
      newPassword: formData.newPassword,
    });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-8" dir="rtl">
      <div className="max-w-2xl mx-auto px-4">
        <Button variant="outline" onClick={() => setLocation("/")} className="mb-6 gap-2">
          <ArrowRight className="w-4 h-4" />
          بازگشت
        </Button>

        <Card className="bg-white">
          <CardHeader>
            <div className="flex items-center gap-3">
              <Lock className="w-6 h-6 text-indigo-600" />
              <div>
                <CardTitle>تغییر رمز عبور</CardTitle>
                <CardDescription>رمز عبور خود را تغییر دهید</CardDescription>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-6">
              <div>
                <Label htmlFor="currentPassword">رمز عبور فعلی *</Label>
                <Input
                  id="currentPassword"
                  type="password"
                  value={formData.currentPassword}
                  onChange={(e) => setFormData({ ...formData, currentPassword: e.target.value })}
                  placeholder="رمز عبور فعلی خود را وارد کنید"
                  required
                />
              </div>

              <div>
                <Label htmlFor="newPassword">رمز عبور جدید *</Label>
                <Input
                  id="newPassword"
                  type="password"
                  value={formData.newPassword}
                  onChange={(e) => setFormData({ ...formData, newPassword: e.target.value })}
                  placeholder="حداقل 6 کاراکتر"
                  required
                />
                <p className="text-xs text-gray-500 mt-1">رمز عبور باید حداقل 6 کاراکتر باشد</p>
              </div>

              <div>
                <Label htmlFor="confirmPassword">تکرار رمز عبور جدید *</Label>
                <Input
                  id="confirmPassword"
                  type="password"
                  value={formData.confirmPassword}
                  onChange={(e) => setFormData({ ...formData, confirmPassword: e.target.value })}
                  placeholder="رمز عبور جدید را دوباره وارد کنید"
                  required
                />
              </div>

              <Button
                type="submit"
                disabled={changePasswordMutation.isPending}
                className="w-full bg-indigo-600 hover:bg-indigo-700 text-white"
              >
                {changePasswordMutation.isPending ? (
                  <>
                    <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                    در حال تغییر...
                  </>
                ) : (
                  "تغییر رمز عبور"
                )}
              </Button>
            </form>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}


