/**
 * API Client for Laravel Backend
 */
import axios, { AxiosError } from 'axios';
import { toast } from 'sonner';

const api = axios.create({
  baseURL: '/api',
  withCredentials: true, // برای ارسال cookies
  headers: {
    'Accept': 'application/json',
    'Content-Type': 'application/json',
  },
});

// Interceptor for errors
api.interceptors.response.use(
  (response) => response,
  (error: AxiosError<any>) => {
    if (error.response?.status === 401) {
      // Redirect to login if unauthorized
      if (window.location.pathname !== '/login') {
        window.location.href = '/login';
      }
    } else if (error.response?.data) {
      // Handle Laravel validation errors
      const data = error.response.data;
      if (data.message) {
        toast.error(data.message);
      } else if (data.errors) {
        // Validation errors
        const errors = Object.values(data.errors).flat();
        errors.forEach((msg: any) => {
          toast.error(msg);
        });
      }
    }
    return Promise.reject(error);
  }
);

// Auth API
export const authApi = {
  login: (username: string, password: string) =>
    api.post('/auth/login', { username, password }),
  me: () => api.get('/auth/me'),
  logout: () => api.post('/auth/logout'),
};

// Advertisements API
export const advertisementsApi = {
  list: () => api.get('/advertisements'),
  getById: (id: number) => api.get(`/advertisements/${id}`),
  create: (data: any) => api.post('/advertisements', data),
};

// Receipts API
export const receiptsApi = {
  getByAdvertisement: (advertisementId: number) =>
    api.get(`/receipts/advertisement/${advertisementId}`),
  create: (data: any) => api.post('/receipts', data),
};

// Reports API
export const reportsApi = {
  list: () => api.get('/reports'),
  getByDateRange: (startDate: string, endDate: string) =>
    api.get('/reports/date-range', { params: { startDate, endDate } }),
  create: (data: any) => api.post('/reports', data),
};

// Users API (Admin only)
export const usersApi = {
  list: () => api.get('/users'),
  create: (data: any) => api.post('/users', data),
  update: (id: number, data: any) => api.put(`/users/${id}`, data),
  updatePassword: (id: number, newPassword: string) =>
    api.put(`/users/${id}/password`, { new_password: newPassword }),
  delete: (id: number) => api.delete(`/users/${id}`),
  changeMyPassword: (currentPassword: string, newPassword: string) =>
    api.post('/users/change-password', { current_password: currentPassword, new_password: newPassword }),
};

export default api;


