<?php

namespace App\Http\Controllers;

use App\Models\Advertisement;
use App\Models\PrintDate;
use App\Models\ReceiptCounter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AdvertisementController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:sanctum');
    }

    public function index()
    {
        $advertisements = Advertisement::where('user_id', Auth::id())
            ->with(['printDates', 'receipts'])
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json($advertisements);
    }

    public function show($id)
    {
        $advertisement = Advertisement::where('id', $id)
            ->where('user_id', Auth::id())
            ->with(['printDates', 'receipts', 'user'])
            ->firstOrFail();

        return response()->json($advertisement);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'issue_date' => 'required|string',
            'advertiser_name' => 'required|string',
            'amount_in_words' => 'required|string',
            'amount_in_numbers' => 'required|integer|min:1',
            'number_of_times' => 'required|integer|min:1',
            'subject' => 'required|string',
            'order_number' => 'nullable|string',
            'signature_location' => 'nullable|string',
            'payment_method' => 'required|in:نقدی,چک,رسید',
            'print_dates' => 'required|array|min:1',
            'print_dates.*' => 'required|string',
        ]);

        DB::beginTransaction();
        try {
            // Generate receipt number
            $receiptNumber = $this->generateReceiptNumber();

            $advertisement = Advertisement::create([
                'receipt_number' => $receiptNumber,
                'issue_date' => $validated['issue_date'],
                'advertiser_name' => $validated['advertiser_name'],
                'amount_in_words' => $validated['amount_in_words'],
                'amount_in_numbers' => $validated['amount_in_numbers'],
                'number_of_times' => $validated['number_of_times'],
                'subject' => $validated['subject'],
                'order_number' => $validated['order_number'] ?? null,
                'signature_location' => $validated['signature_location'] ?? null,
                'payment_method' => $validated['payment_method'],
                'user_id' => Auth::id(),
            ]);

            // Add print dates
            foreach ($validated['print_dates'] as $date) {
                if (!empty($date)) {
                    PrintDate::create([
                        'advertisement_id' => $advertisement->id,
                        'print_date' => $date,
                    ]);
                }
            }

            DB::commit();

            $advertisement->load(['printDates', 'receipts']);

            return response()->json($advertisement, 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'خطا در ایجاد آگهی'], 500);
        }
    }

    private function generateReceiptNumber(): string
    {
        $currentYear = now()->year;
        
        $counter = ReceiptCounter::where('year', $currentYear)->first();
        
        if ($counter) {
            $counter->increment('last_number');
            $nextNumber = $counter->last_number;
        } else {
            $counter = ReceiptCounter::create([
                'last_number' => 1,
                'year' => $currentYear,
            ]);
            $nextNumber = 1;
        }

        return sprintf('%d-%05d', $currentYear, $nextNumber);
    }
}


