# راهنمای تبدیل Frontend از tRPC به Laravel API

## تغییرات اصلی

### 1. حذف tRPC

فایل `client/src/lib/trpc.ts` را حذف کنید و به جای آن از `axios` یا `fetch` استفاده کنید.

### 2. ایجاد API Client

یک فایل جدید `client/src/lib/api.ts` ایجاد کنید:

```typescript
import axios from 'axios';

const api = axios.create({
  baseURL: '/api',
  withCredentials: true, // برای ارسال cookies
  headers: {
    'Accept': 'application/json',
    'Content-Type': 'application/json',
  },
});

export default api;
```

### 3. تبدیل API Calls

#### قبل (tRPC):
```typescript
const { data } = trpc.auth.login.useMutation();
const { data: user } = trpc.auth.me.useQuery();
```

#### بعد (Laravel API):
```typescript
import api from '@/lib/api';

// Login
const login = async (username: string, password: string) => {
  const response = await api.post('/auth/login', { username, password });
  return response.data;
};

// Get current user
const getMe = async () => {
  const response = await api.get('/auth/me');
  return response.data.user;
};
```

### 4. تبدیل تمام Endpoints

#### Authentication
- `trpc.auth.login.useMutation()` → `api.post('/auth/login', data)`
- `trpc.auth.me.useQuery()` → `api.get('/auth/me')`
- `trpc.auth.logout.useMutation()` → `api.post('/auth/logout')`

#### Advertisements
- `trpc.advertisements.list.useQuery()` → `api.get('/advertisements')`
- `trpc.advertisements.getById.useQuery({ id })` → `api.get(`/advertisements/${id}`)`
- `trpc.advertisements.create.useMutation()` → `api.post('/advertisements', data)`

#### Receipts
- `trpc.receipts.getByAdvertisement.useQuery({ advertisementId })` → `api.get(`/receipts/advertisement/${advertisementId}`)`
- `trpc.receipts.create.useMutation()` → `api.post('/receipts', data)`

#### Reports
- `trpc.reports.list.useQuery()` → `api.get('/reports')`
- `trpc.reports.getByDateRange.useQuery({ startDate, endDate })` → `api.get('/reports/date-range', { params: { startDate, endDate } })`
- `trpc.reports.create.useMutation()` → `api.post('/reports', data)`

#### Users (Admin)
- `trpc.users.list.useQuery()` → `api.get('/users')`
- `trpc.users.create.useMutation()` → `api.post('/users', data)`
- `trpc.users.update.useMutation()` → `api.put(`/users/${id}`, data)`
- `trpc.users.updatePassword.useMutation()` → `api.put(`/users/${id}/password`, data)`
- `trpc.users.delete.useMutation()` → `api.delete(`/users/${id}`)`
- `trpc.users.changeMyPassword.useMutation()` → `api.post('/users/change-password', data)`

### 5. تبدیل useAuth Hook

فایل `client/src/_core/hooks/useAuth.ts` را به این صورت تغییر دهید:

```typescript
import { useState, useEffect } from 'react';
import api from '@/lib/api';

export function useAuth() {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    checkAuth();
  }, []);

  const checkAuth = async () => {
    try {
      const response = await api.get('/auth/me');
      setUser(response.data.user);
    } catch (error) {
      setUser(null);
    } finally {
      setLoading(false);
    }
  };

  const login = async (username: string, password: string) => {
    const response = await api.post('/auth/login', { username, password });
    setUser(response.data.user);
    return response.data.user;
  };

  const logout = async () => {
    await api.post('/auth/logout');
    setUser(null);
  };

  return {
    user,
    loading,
    isAuthenticated: !!user,
    login,
    logout,
    refresh: checkAuth,
  };
}
```

### 6. تبدیل صفحات

#### Login.tsx
```typescript
const loginMutation = useMutation({
  mutationFn: (data: { username: string; password: string }) =>
    api.post('/auth/login', data),
  onSuccess: (response) => {
    setUser(response.data.user);
    toast.success('ورود با موفقیت انجام شد');
    navigate('/');
  },
});
```

#### CreateAdvertisement.tsx
```typescript
const createMutation = useMutation({
  mutationFn: (data: any) => api.post('/advertisements', data),
  onSuccess: () => {
    toast.success('آگهی با موفقیت ایجاد شد');
    navigate('/');
  },
});
```

### 7. Error Handling

Laravel errors را به این صورت handle کنید:

```typescript
try {
  await api.post('/auth/login', data);
} catch (error: any) {
  if (error.response?.data?.message) {
    toast.error(error.response.data.message);
  } else if (error.response?.data?.errors) {
    // Validation errors
    const errors = error.response.data.errors;
    Object.values(errors).flat().forEach((msg: any) => {
      toast.error(msg);
    });
  } else {
    toast.error('خطایی رخ داد');
  }
}
```

### 8. Jalaali Helper

Helper Jalaali در Laravel موجود است اما در frontend می‌توانید از همان `jalaali-js` استفاده کنید.

## نکات مهم

1. تمام درخواست‌ها باید `withCredentials: true` داشته باشند
2. Header `Accept: application/json` باید ارسال شود
3. Errors از Laravel به صورت `{ message: string }` یا `{ errors: object }` برمی‌گردند
4. Response structure: `{ user: {...} }` یا `{ data: [...] }`

## مثال کامل

```typescript
// api.ts
import axios from 'axios';

const api = axios.create({
  baseURL: '/api',
  withCredentials: true,
  headers: {
    'Accept': 'application/json',
    'Content-Type': 'application/json',
  },
});

// Interceptor for errors
api.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response?.status === 401) {
      // Redirect to login
      window.location.href = '/login';
    }
    return Promise.reject(error);
  }
);

export default api;
```


