import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { useState } from "react";
import { Loader2 } from "lucide-react";
import { toast } from "sonner";
import { APP_TITLE } from "@/const";
import { authApi } from "@/lib/api";
import { useAuth } from "@/_core/hooks/useAuth";

export default function Login() {
  const [, setLocation] = useLocation();
  const { refresh } = useAuth();
  const [formData, setFormData] = useState({
    username: "",
    password: "",
  });

  const loginMutation = useMutation({
    mutationFn: (data: { username: string; password: string }) =>
      authApi.login(data.username, data.password),
    onSuccess: async (response) => {
      await refresh();
      toast.success("ورود با موفقیت انجام شد");
      setLocation("/");
      window.location.reload();
    },
    onError: (error: any) => {
      const message = error.response?.data?.message || error.response?.data?.errors?.username?.[0] || "خطا در ورود";
      toast.error(message);
    },
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.username.trim()) {
      toast.error("نام کاربری الزامی است");
      return;
    }

    if (!formData.password.trim()) {
      toast.error("رمز عبور الزامی است");
      return;
    }

    loginMutation.mutate({
      username: formData.username,
      password: formData.password,
    });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 flex items-center justify-center" dir="rtl">
      <div className="max-w-md w-full mx-auto px-4">
        <Card className="bg-white shadow-lg">
          <CardHeader className="text-center">
            <CardTitle className="text-2xl font-bold">{APP_TITLE}</CardTitle>
            <CardDescription>لطفاً وارد حساب کاربری خود شوید</CardDescription>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div>
                <Label htmlFor="username">نام کاربری</Label>
                <Input
                  id="username"
                  type="text"
                  value={formData.username}
                  onChange={(e) => setFormData((prev) => ({ ...prev, username: e.target.value }))}
                  placeholder="نام کاربری خود را وارد کنید"
                  required
                  autoFocus
                />
              </div>

              <div>
                <Label htmlFor="password">رمز عبور</Label>
                <Input
                  id="password"
                  type="password"
                  value={formData.password}
                  onChange={(e) => setFormData((prev) => ({ ...prev, password: e.target.value }))}
                  placeholder="رمز عبور خود را وارد کنید"
                  required
                />
              </div>

              <Button
                type="submit"
                disabled={loginMutation.isPending}
                className="w-full bg-indigo-600 hover:bg-indigo-700 text-white"
              >
                {loginMutation.isPending ? (
                  <>
                    <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                    در حال ورود...
                  </>
                ) : (
                  "ورود"
                )}
              </Button>
            </form>

            <div className="mt-4 text-center text-sm text-gray-500">
              <p>حساب کاربری ندارید؟ با مدیر سیستم تماس بگیرید.</p>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}


