import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { APP_LOGO, APP_TITLE, getLoginUrl } from "@/const";
import { advertisementsApi } from "@/lib/api";
import { Plus, FileText, BarChart3, Loader2 } from "lucide-react";
import { useLocation } from "wouter";
import { useEffect } from "react";
import { useQuery } from "@tanstack/react-query";

export default function Home() {
  const { user, loading: authLoading, isAuthenticated } = useAuth();
  const [, setLocation] = useLocation();
  
  const { data: advertisements, isLoading: adsLoading } = useQuery({
    queryKey: ['advertisements'],
    queryFn: async () => {
      const response = await advertisementsApi.list();
      return response.data;
    },
    enabled: isAuthenticated,
  });

  useEffect(() => {
    if (!authLoading && !isAuthenticated) {
      setLocation(getLoginUrl());
    }
  }, [authLoading, isAuthenticated, setLocation]);

  if (authLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100">
        <div className="text-center">
          <Loader2 className="w-12 h-12 animate-spin text-indigo-600 mx-auto mb-4" />
          <p className="text-gray-600">در حال بارگذاری...</p>
        </div>
      </div>
    );
  }

  if (!isAuthenticated) {
    return null;
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100" dir="rtl">
      {/* Header */}
      <header className="bg-white shadow-sm border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4 flex items-center justify-between flex-row-reverse">
          <div className="flex items-center gap-3 flex-row-reverse">
            <img src={APP_LOGO} alt={APP_TITLE} className="w-10 h-10" />
            <div className="text-right">
              <h1 className="text-2xl font-bold text-gray-900">{APP_TITLE}</h1>
              <p className="text-sm text-gray-500">مدیریت آگهی‌ها و قبض‌ها</p>
            </div>
          </div>
          <div className="flex items-center gap-4">
            <span className="text-sm text-gray-600">خوش آمدید, {user?.name || user?.username}</span>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setLocation("/change-password")}
            >
              تغییر رمز عبور
            </Button>
            {user?.role === "admin" && (
              <Button
                variant="outline"
                size="sm"
                onClick={() => setLocation("/users")}
              >
                مدیریت کاربران
              </Button>
            )}
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Quick Actions */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
          <Button
            onClick={() => setLocation("/advertisements/create")}
            className="h-auto py-4 bg-indigo-600 hover:bg-indigo-700 text-white rounded-lg flex flex-col items-center gap-2"
          >
            <Plus className="w-6 h-6" />
            <span>آگهی جدید</span>
          </Button>
          <Button
            onClick={() => setLocation("/reports")}
            className="h-auto py-4 bg-blue-600 hover:bg-blue-700 text-white rounded-lg flex flex-col items-center gap-2"
          >
            <FileText className="w-6 h-6" />
            <span>قبض‌ها</span>
          </Button>
          <Button
            onClick={() => setLocation("/reports")}
            className="h-auto py-4 bg-green-600 hover:bg-green-700 text-white rounded-lg flex flex-col items-center gap-2"
          >
            <BarChart3 className="w-6 h-6" />
            <span>گزارش‌ها</span>
          </Button>
        </div>

        {/* Recent Advertisements */}
        <Card className="bg-white">
          <CardHeader>
            <CardTitle>آگهی‌های اخیر</CardTitle>
            <CardDescription>آخرین آگهی‌های ثبت شده</CardDescription>
          </CardHeader>
          <CardContent>
            {adsLoading ? (
              <div className="flex justify-center py-8">
                <Loader2 className="w-6 h-6 animate-spin text-indigo-600" />
              </div>
            ) : advertisements && advertisements.length > 0 ? (
              <div className="overflow-x-auto">
                <table className="w-full text-sm">
                  <thead className="border-b border-gray-200">
                    <tr className="text-right">
                      <th className="px-4 py-2 font-semibold text-gray-700">شماره قبض</th>
                      <th className="px-4 py-2 font-semibold text-gray-700">نام صاحب اگهی</th>
                      <th className="px-4 py-2 font-semibold text-gray-700">موضوع</th>
                      <th className="px-4 py-2 font-semibold text-gray-700">مبلغ</th>
                      <th className="px-4 py-2 font-semibold text-gray-700">تاریخ</th>
                      <th className="px-4 py-2 font-semibold text-gray-700">عملیات</th>
                    </tr>
                  </thead>
                  <tbody>
                    {advertisements.slice(0, 5).map((ad: any) => (
                      <tr key={ad.id} className="border-b border-gray-100 hover:bg-gray-50">
                        <td className="px-4 py-3 font-mono text-red-600 font-bold">{ad.receipt_number}</td>
                        <td className="px-4 py-3">{ad.advertiser_name}</td>
                        <td className="px-4 py-3 text-gray-600">{ad.subject}</td>
                        <td className="px-4 py-3 font-semibold">{ad.amount_in_numbers.toLocaleString()} ریال</td>
                        <td className="px-4 py-3 text-gray-600">{ad.issue_date}</td>
                        <td className="px-4 py-3 flex gap-2">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => setLocation(`/advertisements/${ad.id}`)}
                          >
                            جزئیات
                          </Button>
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => setLocation(`/receipts/${ad.id}`)}
                          >
                            قبض
                          </Button>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            ) : (
              <div className="text-center py-8">
                <p className="text-gray-500">هیچ آگهی‌ای ثبت نشده است</p>
                <Button
                  onClick={() => setLocation("/advertisements/create")}
                  className="mt-4"
                >
                  ایجاد آگهی جدید
                </Button>
              </div>
            )}
          </CardContent>
        </Card>
      </main>
    </div>
  );
}


