import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { advertisementsApi } from "@/lib/api";
import { useLocation } from "wouter";
import { useState } from "react";
import { Loader2, Plus, X } from "lucide-react";
import { toast } from "sonner";
import { jalaliToGregorian } from "@/lib/dateUtils";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { todayJalali } from "@/lib/jalali";

export default function CreateAdvertisement() {
  const { isAuthenticated } = useAuth();
  const [, setLocation] = useLocation();
  const queryClient = useQueryClient();

  const [formData, setFormData] = useState({
    issueDate: todayJalali(),
    advertiserName: "",
    amountInWords: "",
    amountInNumbers: 0,
    numberOfTimes: 1,
    subject: "",
    orderNumber: "",
    signatureLocation: "",
    paymentMethod: "نقدی" as const,
    printDates: [""],
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => advertisementsApi.create(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['advertisements'] });
      toast.success("آگهی با موفقیت ایجاد شد");
      setLocation("/");
    },
    onError: (error: any) => {
      const message = error.response?.data?.message || "خطایی در ایجاد آگهی رخ داد";
      toast.error(message);
    },
  });

  if (!isAuthenticated) {
    return null;
  }

  const handleInputChange = (field: string, value: any) => {
    setFormData((prev) => ({
      ...prev,
      [field]: value,
    }));
  };

  const handlePrintDateChange = (index: number, value: string) => {
    const newDates = [...formData.printDates];
    newDates[index] = value;
    setFormData((prev) => ({
      ...prev,
      printDates: newDates,
    }));
  };

  const addPrintDate = () => {
    if (formData.printDates.length < 3) {
      setFormData((prev) => ({
        ...prev,
        printDates: [...prev.printDates, ""],
      }));
    }
  };

  const removePrintDate = (index: number) => {
    setFormData((prev) => ({
      ...prev,
      printDates: prev.printDates.filter((_, i) => i !== index),
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.advertiserName.trim()) {
      toast.error("نام صاحب اگهی الزامی است");
      return;
    }

    if (!formData.subject.trim()) {
      toast.error("موضوع اگهی الزامی است");
      return;
    }

    if (formData.amountInNumbers <= 0) {
      toast.error("مبلغ باید بیشتر از صفر باشد");
      return;
    }

    const validDates = formData.printDates.filter((d) => d.trim());
    if (validDates.length === 0) {
      toast.error("حداقل یک تاریخ چاپ الزامی است");
      return;
    }

    if (validDates.length > 3) {
      toast.error("حداکثر 3 تاریخ چاپ مجاز است");
      return;
    }

    const gregorianIssueDate = jalaliToGregorian(formData.issueDate);
    const gregorianPrintDates = validDates.map(date => jalaliToGregorian(date)).filter(Boolean);

    if (!gregorianIssueDate) {
      toast.error("تاریخ صدور معتبر نیست");
      return;
    }

    createMutation.mutate({
      issue_date: gregorianIssueDate,
      advertiser_name: formData.advertiserName,
      amount_in_words: formData.amountInWords,
      amount_in_numbers: formData.amountInNumbers,
      number_of_times: formData.numberOfTimes,
      subject: formData.subject,
      order_number: formData.orderNumber || null,
      signature_location: formData.signatureLocation || null,
      payment_method: formData.paymentMethod,
      print_dates: gregorianPrintDates,
    });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-8" dir="rtl">
      <div className="max-w-2xl mx-auto px-4">
        <Button
          variant="outline"
          onClick={() => setLocation("/")}
          className="mb-6"
        >
          ← بازگشت
        </Button>

        <Card>
          <CardHeader className="bg-gradient-to-r from-indigo-500 to-blue-500 text-white rounded-t-lg">
            <CardTitle>ایجاد آگهی جدید</CardTitle>
            <CardDescription className="text-indigo-100">
              اطلاعات آگهی و تاریخ‌های چاپ را وارد کنید
            </CardDescription>
          </CardHeader>

          <CardContent className="py-6">
            <form onSubmit={handleSubmit} className="space-y-6">
              <div>
                <Label htmlFor="issueDate">تاریخ صدور (شمسی)</Label>
                <Input
                  id="issueDate"
                  type="text"
                  placeholder="YYYY-MM-DD"
                  value={formData.issueDate}
                  onChange={(e) => handleInputChange("issueDate", e.target.value)}
                  required
                  dir="ltr"
                />
                <p className="text-xs text-gray-500 mt-1">
                  به فرمت YYYY-MM-DD وارد کنید (مثال: 1403-09-05)
                </p>
              </div>

              <div>
                <Label htmlFor="advertiserName">نام صاحب اگهی</Label>
                <Input
                  id="advertiserName"
                  type="text"
                  value={formData.advertiserName}
                  onChange={(e) => handleInputChange("advertiserName", e.target.value)}
                  required
                />
              </div>

              <div>
                <Label htmlFor="subject">موضوع اگهی</Label>
                <Textarea
                  id="subject"
                  value={formData.subject}
                  onChange={(e) => handleInputChange("subject", e.target.value)}
                  required
                  rows={3}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="amountInNumbers">مبلغ به عدد (ریال)</Label>
                  <Input
                    id="amountInNumbers"
                    type="number"
                    value={formData.amountInNumbers}
                    onChange={(e) => handleInputChange("amountInNumbers", parseInt(e.target.value) || 0)}
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="amountInWords">مبلغ به حروف</Label>
                  <Input
                    id="amountInWords"
                    type="text"
                    value={formData.amountInWords}
                    onChange={(e) => handleInputChange("amountInWords", e.target.value)}
                    required
                  />
                </div>
              </div>

              <div>
                <Label htmlFor="numberOfTimes">تعداد نوبت</Label>
                <Select
                  value={formData.numberOfTimes.toString()}
                  onValueChange={(value) => handleInputChange("numberOfTimes", parseInt(value))}
                >
                  <SelectTrigger id="numberOfTimes">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1</SelectItem>
                    <SelectItem value="2">2</SelectItem>
                    <SelectItem value="3">3</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="paymentMethod">نحوه وصول</Label>
                <Select
                  value={formData.paymentMethod}
                  onValueChange={(value) => handleInputChange("paymentMethod", value)}
                >
                  <SelectTrigger id="paymentMethod">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="نقدی">نقدی</SelectItem>
                    <SelectItem value="چک">چک</SelectItem>
                    <SelectItem value="رسید">رسید</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-3">
                <Label>تاریخ‌های چاپ (شمسی)</Label>
                {formData.printDates.map((date, index) => (
                  <div key={index} className="flex gap-2">
                    <Input
                      type="text"
                      placeholder="YYYY-MM-DD"
                      value={date}
                      onChange={(e) => handlePrintDateChange(index, e.target.value)}
                      dir="ltr"
                      required={index === 0}
                    />
                    {formData.printDates.length > 1 && (
                      <Button
                        type="button"
                        variant="destructive"
                        size="sm"
                        onClick={() => removePrintDate(index)}
                      >
                        <X className="w-4 h-4" />
                      </Button>
                    )}
                  </div>
                ))}
                {formData.printDates.length < 3 && (
                  <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={addPrintDate}
                  >
                    <Plus className="w-4 h-4 ml-2" />
                    اضافه کردن تاریخ چاپ
                  </Button>
                )}
                <p className="text-xs text-gray-500">
                  به فرمت YYYY-MM-DD وارد کنید (مثال: 1403-09-05)
                </p>
              </div>

              <div>
                <Label htmlFor="orderNumber">شماره دستور (اختیاری)</Label>
                <Input
                  id="orderNumber"
                  type="text"
                  value={formData.orderNumber}
                  onChange={(e) => handleInputChange("orderNumber", e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="signatureLocation">جای امضا (اختیاری)</Label>
                <Input
                  id="signatureLocation"
                  type="text"
                  value={formData.signatureLocation}
                  onChange={(e) => handleInputChange("signatureLocation", e.target.value)}
                />
              </div>

              <Button
                type="submit"
                className="w-full bg-indigo-600 hover:bg-indigo-700 text-white py-2"
                disabled={createMutation.isPending}
              >
                {createMutation.isPending ? (
                  <>
                    <Loader2 className="w-4 h-4 ml-2 animate-spin" />
                    در حال ایجاد...
                  </>
                ) : (
                  "ایجاد آگهی"
                )}
              </Button>
            </form>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}


