import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { advertisementsApi, receiptsApi } from "@/lib/api";
import { useLocation } from "wouter";
import { Loader2, Download, Printer } from "lucide-react";
import { toast } from "sonner";
import { useQuery } from "@tanstack/react-query";
import { gregorianToJalali } from "@/lib/dateUtils";

interface AdvertisementDetailProps {
  id: number;
}

export default function AdvertisementDetail({ id }: AdvertisementDetailProps) {
  const { isAuthenticated } = useAuth();
  const [, setLocation] = useLocation();

  const { data: advertisement, isLoading: adLoading } = useQuery({
    queryKey: ['advertisements', id],
    queryFn: async () => {
      const response = await advertisementsApi.getById(id);
      return response.data;
    },
    enabled: isAuthenticated,
  });

  const { data: receiptData, isLoading: datesLoading } = useQuery({
    queryKey: ['receipts', 'advertisement', id],
    queryFn: async () => {
      const response = await receiptsApi.getByAdvertisement(id);
      return response.data;
    },
    enabled: isAuthenticated && !!advertisement,
  });

  if (!isAuthenticated) {
    return null;
  }

  if (adLoading || datesLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100">
        <Loader2 className="w-12 h-12 animate-spin text-indigo-600" />
      </div>
    );
  }

  if (!advertisement) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-8">
        <div className="max-w-2xl mx-auto px-4">
          <Button variant="outline" onClick={() => setLocation("/")} className="mb-6">
            ← بازگشت
          </Button>
          <Card className="bg-white">
            <CardContent className="py-8 text-center">
              <p className="text-gray-500">آگهی مورد نظر یافت نشد</p>
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  const printDates = receiptData?.print_dates || [];
  const issueDateJalali = advertisement.issue_date ? gregorianToJalali(advertisement.issue_date) : advertisement.issue_date;

  const handlePrint = () => {
    window.print();
  };

  const handleDownload = async () => {
    toast.promise(
      new Promise((resolve) => {
        setTimeout(() => {
          const element = document.getElementById("receipt-content");
          if (element) {
            const printWindow = window.open("", "", "width=800,height=600");
            if (printWindow) {
              printWindow.document.write(element.innerHTML);
              printWindow.document.close();
              printWindow.print();
            }
          }
          resolve(true);
        }, 500);
      }),
      {
        loading: "در حال آماده‌سازی...",
        success: "آماده برای چاپ",
        error: "خطایی رخ داد",
      }
    );
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-8" dir="rtl">
      <div className="max-w-4xl mx-auto px-4">
        <div className="flex gap-2 mb-6">
          <Button variant="outline" onClick={() => setLocation("/")}>
            ← بازگشت
          </Button>
          <Button onClick={handlePrint} className="bg-blue-600 hover:bg-blue-700">
            <Printer className="w-4 h-4 ml-2" />
            چاپ
          </Button>
          <Button onClick={handleDownload} className="bg-green-600 hover:bg-green-700">
            <Download className="w-4 h-4 ml-2" />
            دانلود
          </Button>
        </div>

        <Card className="bg-white" id="receipt-content">
          <CardHeader className="border-b border-gray-200">
            <div className="flex justify-between items-start">
              <div>
                <CardTitle className="text-2xl">قبض حق‌الدرج آگهی</CardTitle>
                <CardDescription>روزنامه خبر جنوب</CardDescription>
              </div>
              <div className="text-right">
                <p className="text-red-600 font-bold text-lg">شماره قبض: {advertisement.receipt_number}</p>
                <p className="text-gray-600 text-sm">تاریخ: {issueDateJalali}</p>
              </div>
            </div>
          </CardHeader>

          <CardContent className="py-6">
            <div className="grid grid-cols-2 gap-6 mb-6">
              <div className="text-right">
                <p className="text-sm text-gray-600">نام صاحب اگهی</p>
                <p className="font-semibold text-lg">{advertisement.advertiser_name}</p>
              </div>
              <div className="text-right">
                <p className="text-sm text-gray-600">موضوع اگهی</p>
                <p className="font-semibold text-lg">{advertisement.subject}</p>
              </div>
            </div>

            <div className="grid grid-cols-2 gap-6 mb-6">
              <div className="text-right">
                <p className="text-sm text-gray-600">مبلغ به عدد</p>
                <p className="font-semibold text-lg">{advertisement.amount_in_numbers.toLocaleString()} ریال</p>
              </div>
              <div className="text-right">
                <p className="text-sm text-gray-600">مبلغ به حروف</p>
                <p className="font-semibold text-lg">{advertisement.amount_in_words}</p>
              </div>
            </div>

            <div className="grid grid-cols-2 gap-6 mb-6">
              <div className="text-right">
                <p className="text-sm text-gray-600">تعداد نوبت</p>
                <p className="font-semibold text-lg">{advertisement.number_of_times}</p>
              </div>
              <div className="text-right">
                <p className="text-sm text-gray-600">نحوه وصول</p>
                <p className="font-semibold text-lg">{advertisement.payment_method}</p>
              </div>
            </div>

            {advertisement.order_number && (
              <div className="mb-6">
                <p className="text-sm text-gray-600">شماره دستور</p>
                <p className="font-semibold">{advertisement.order_number}</p>
              </div>
            )}

            {printDates.length > 0 && (
              <div className="mb-6">
                <p className="text-sm text-gray-600 mb-2">تاریخ‌های چاپ</p>
                <div className="flex gap-2 flex-wrap">
                  {printDates.map((date: any, index: number) => (
                    <span key={index} className="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm">
                      {date.print_date ? gregorianToJalali(date.print_date) : date.print_date}
                    </span>
                  ))}
                </div>
              </div>
            )}

            {advertisement.signature_location && (
              <div className="border-t border-gray-200 pt-6 mt-6">
                <p className="text-sm text-gray-600 mb-4">جای امضا</p>
                <div className="h-20 border border-gray-300 rounded p-4 text-gray-400 text-sm">
                  {advertisement.signature_location}
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}

