<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:sanctum');
    }

    public function index()
    {
        if (!Auth::user()->isAdmin()) {
            return response()->json(['error' => 'دسترسی غیرمجاز'], 403);
        }

        $users = User::select('id', 'username', 'name', 'email', 'role', 'created_at', 'last_signed_in')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json($users);
    }

    public function store(Request $request)
    {
        if (!Auth::user()->isAdmin()) {
            return response()->json(['error' => 'دسترسی غیرمجاز'], 403);
        }

        $validated = $request->validate([
            'username' => 'required|string|max:64|unique:users,username',
            'password' => 'required|string|min:6',
            'name' => 'nullable|string',
            'email' => 'nullable|email|max:320',
            'role' => ['required', Rule::in(['user', 'admin'])],
        ]);

        $user = User::create([
            'username' => $validated['username'],
            'password' => Hash::make($validated['password']),
            'name' => $validated['name'] ?? null,
            'email' => $validated['email'] ?? null,
            'role' => $validated['role'],
        ]);

        return response()->json(['success' => true], 201);
    }

    public function update(Request $request, $id)
    {
        if (!Auth::user()->isAdmin()) {
            return response()->json(['error' => 'دسترسی غیرمجاز'], 403);
        }

        $validated = $request->validate([
            'name' => 'nullable|string',
            'email' => 'nullable|email|max:320',
            'role' => [Rule::in(['user', 'admin'])],
        ]);

        $user = User::findOrFail($id);
        $user->update($validated);

        return response()->json(['success' => true]);
    }

    public function updatePassword(Request $request, $id)
    {
        if (!Auth::user()->isAdmin()) {
            return response()->json(['error' => 'دسترسی غیرمجاز'], 403);
        }

        $validated = $request->validate([
            'new_password' => 'required|string|min:6',
        ]);

        $user = User::findOrFail($id);
        $user->update([
            'password' => Hash::make($validated['new_password']),
        ]);

        return response()->json(['success' => true]);
    }

    public function changeMyPassword(Request $request)
    {
        $validated = $request->validate([
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:6',
        ]);

        $user = Auth::user();

        if (!Hash::check($validated['current_password'], $user->password)) {
            return response()->json(['error' => 'رمز عبور فعلی اشتباه است'], 400);
        }

        $user->update([
            'password' => Hash::make($validated['new_password']),
        ]);

        return response()->json(['success' => true]);
    }

    public function destroy($id)
    {
        if (!Auth::user()->isAdmin()) {
            return response()->json(['error' => 'دسترسی غیرمجاز'], 403);
        }

        if ($id == Auth::id()) {
            return response()->json(['error' => 'نمی‌توانید حساب کاربری خود را حذف کنید'], 400);
        }

        $user = User::findOrFail($id);
        $user->delete();

        return response()->json(['success' => true]);
    }
}

